!-------------------------------------------------------------------------------

! This file is part of code_saturne, a general-purpose CFD tool.
!
! Copyright (C) 1998-2024 EDF S.A.
!
! This program is free software; you can redistribute it and/or modify it under
! the terms of the GNU General Public License as published by the Free Software
! Foundation; either version 2 of the License, or (at your option) any later
! version.
!
! This program is distributed in the hope that it will be useful, but WITHOUT
! ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
! details.
!
! You should have received a copy of the GNU General Public License along with
! this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
! Street, Fifth Floor, Boston, MA 02110-1301, USA.

!-------------------------------------------------------------------------------

!> \file ppthch.f90
!> Module for specific physics thermophysical data

module ppthch

  !===========================================================================

  use, intrinsic :: iso_c_binding

  use cstphy

  implicit none

  !===========================================================================

  !> \defgroup  thermophysical Module for specific physics thermophysical data

  !> \addtogroup thermophysical
  !> \{

  !--> DONNEES

  !> maximal number of global species
  integer    ngazgm

  !> maximal number of elementary gas components
  integer    ngazem

  !> maximal number of tabulation points
  integer    npot

  !> maximal number of atomic species
  integer    natom

  !> maximal number of global reactions in gas phase
  integer    nrgazm

  parameter( ngazgm = 25 , ngazem = 20 ,                                     &
             npot  = 50 , natom  = 5   , nrgazm = 1 )
  integer    iatc, iath, iato, iatn , iats
  parameter( iatc = 1, iath = 2, iato = 3, iatn = 4 , iats = 5 )

  !> number of elementary gas components
  integer, pointer, save ::  ngaze
  !> number of global species
  integer, pointer, save ::  ngazg
  !> number of atomic species
  integer, pointer, save ::  nato

  !> number of global reactions in gas phase
  integer, pointer, save ::  nrgaz

  !> rank of O2 in gas composition
  integer, pointer, save ::  iio2
  !> rank of CO2 in gas composition
  integer, pointer, save ::  iico2
  !> rank of C in gas composition
  integer, pointer, save ::  iic

  !> molar mass of an elementary gas component
  real(c_double), pointer, save ::  wmole(:)

  !> molar mass of a global species
  real(c_double), pointer, save ::  wmolg(:)

  !> molar mass of atoms
  real(c_double), pointer, save ::  wmolat(:)

  !> Mixing rate at the stoichiometry
  real(c_double), pointer, save ::  fs(:)

  !> Absorption coefficient of global species
  real(c_double), pointer, save ::  ckabsg(:)

  !> Molar coefficient of CO2
  real(c_double), pointer, save ::  xco2
  !> Molar coefficient of H2O
  real(c_double), pointer, save ::  xh2o

  !=============================================================================

  !> \}

  !=============================================================================

  interface

    !---------------------------------------------------------------------------

    !> \cond DOXYGEN_SHOULD_SKIP_THIS

    !---------------------------------------------------------------------------

    ! Interface to C function retrieving pointers to members of the
    ! global physical model flags

    subroutine cs_f_ppthch_get_pointers(                                       &
         p_ngaze, p_ngazg, p_nato, p_nrgaz,                                    &
         p_iic, p_iico2, p_iio2,                                               &
         p_wmole, p_wmolg, p_wmolat,                                           &
         p_xco2, p_xh2o, p_fs, p_ckabsg)                                       &
      bind(C, name='cs_f_ppthch_get_pointers')
      use, intrinsic :: iso_c_binding
      implicit none
      type(c_ptr), intent(out) ::                                              &
         p_ngaze, p_ngazg, p_nato, p_nrgaz,                                    &
         p_iic, p_iico2, p_iio2,                                               &
         p_wmole, p_wmolg, p_wmolat,                                           &
         p_xco2, p_xh2o, p_fs, p_ckabsg
    end subroutine cs_f_ppthch_get_pointers

    !---------------------------------------------------------------------------

    !> (DOXYGEN_SHOULD_SKIP_THIS) \endcond

    !---------------------------------------------------------------------------

  end interface

  !=============================================================================

contains

  !=============================================================================

  !> \brief Initialize Fortran combustion models properties API.
  !> This maps Fortran pointers to global C variables.

  subroutine thch_models_init() &
    bind(C, name='cs_f_thch_models_init')

    use, intrinsic :: iso_c_binding
    implicit none

    ! Local variables

    type(c_ptr) ::                                                             &
         p_ngaze, p_ngazg, p_nato, p_nrgaz,                                    &
         p_iic, p_iico2, p_iio2,                                               &
         p_wmole,p_wmolg, p_wmolat,                                            &
         p_xco2, p_xh2o, p_fs, p_ckabsg

    call cs_f_ppthch_get_pointers(                                             &
         p_ngaze, p_ngazg, p_nato, p_nrgaz,                                    &
         p_iic, p_iico2, p_iio2,                                               &
         p_wmole, p_wmolg, p_wmolat,                                           &
         p_xco2, p_xh2o, p_fs, p_ckabsg)

    call c_f_pointer(p_ngaze, ngaze)
    call c_f_pointer(p_ngazg, ngazg)
    call c_f_pointer(p_nato, nato)
    call c_f_pointer(p_nrgaz, nrgaz)
    call c_f_pointer(p_iic, iic)
    call c_f_pointer(p_iico2, iico2)
    call c_f_pointer(p_iio2, iio2)
    call c_f_pointer(p_wmole, wmole, [ngazem])
    call c_f_pointer(p_wmolg, wmolg, [ngazgm])
    call c_f_pointer(p_wmolat, wmolat, [natom])
    call c_f_pointer(p_xco2, xco2)
    call c_f_pointer(p_xh2o, xh2o)
    call c_f_pointer(p_fs, fs, [nrgazm])
    call c_f_pointer(p_ckabsg, ckabsg, [ngazgm])

  end subroutine thch_models_init

  !=============================================================================

end module ppthch
